%%%% A Matlab Code for 2D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT2D(200,60,0.3,3)
function SEMDOT2D(nelx,nely,volfrac,rmin)
nodenrs=reshape(1:(1+nelx)*(1+nely),1+nely,1+nelx);
edofVec=reshape(2*nodenrs(1:end-1,1:end-1)+1,nelx*nely,1);
edofMat=repmat(edofVec,1,8)+repmat([0 1 2*nely+[2 3 0 1] -2 -1],nelx*nely,1);
iK=reshape(kron(edofMat,ones(8,1))',64*nelx*nely,1);
jK=reshape(kron(edofMat,ones(1,8))',64*nelx*nely,1);
%% GEOMETRY & MATERIAL
dx=5; dy=dx; thickness=1;
NU=0.3; E=[7e4;3e3]; density=[2.7e-9;1.2e-9];
[KE1,ME1]=lkm(E(1),NU,density(1),dx,dy,thickness);
[KE2,ME2]=lkm(E(2),NU,density(2),dx,dy,thickness);
%% FILTER
[H,Hs]=HHs(nelx,nely,rmin);
[Hn,Hns]=HnHns(nelx,nely);
%% OPTIMIZATION
x=repmat(volfrac,nely,nelx); xPhys=x; nG=10; ngrid=nG-1;
%% DEFINE SUPPORTS
fixeddofs=[1:2*(nely+1)];
alldofs=[1:2*(nelx+1)*(nely+1)]; freedofs=setdiff(alldofs,fixeddofs);
%% DIFF FREQ
% wt=50*2*pi; scale_obj=1e-6;
% wt=700*2*pi; scale_obj=1e-2;
 wt=1500*2*pi; scale_obj=0.1;
%% HARMONIC
F=zeros(2*(nelx+1)*(nely+1),1);
Amplitude=1e3; F(2*(nelx+1)*(nely+1),1)=-Amplitude;
%% DAMPING
alphac=5.24e-01; betac=1.63e-07;
%% PROJECTION
beta=0.5; ER=0.5;
%% MMA
m=1; n=nelx*nely; xold1=x(:); xold2=xold1;
xmin=zeros(n,1); xmax=ones(n,1); low=xmin; upp=xmax;
a0=1; a=zeros(m,1); cmma=10000*ones(m,1); d=zeros(m,1);
move=0.01; scale_cons=1000;
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey]=meshgrid(0:nelx,0:nely);
[fnx,fny]=meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely);
%% CONVERGENCE
maxiter=1500;
%% SEMODT
Emin=1e-3; iter=0; change=1;
%% START ITERATION
while (change > 0.001 || tol>0.001) && iter < maxiter
    i=sqrt(-1); iter=iter+1; beta=beta+ER;
    Con(iter,1)=sum(xPhys(:))/nelx/nely;
    %% FE-ANALYSIS
    I=ones(nely,nelx);
    sK=reshape((KE1(:)*xPhys(:)'+KE2(:)*I(:)'-KE2(:)*xPhys(:)'),64*nelx*nely,1);
    sM=reshape((ME1(:)*xPhys(:)'+ME2(:)*I(:)'-ME2(:)*xPhys(:)'),64*nelx*nely,1);
    K=sparse(iK,jK,sK); K=(K+K')/2;
    M=sparse(iK,jK,sM); M=(M+M')/2;
    C=alphac*M+betac*K; C=(C+C')/2;
    G=K-wt*wt*M+i*wt*C;
    U=zeros(2*(nelx+1)*(nely+1),1);
    U(freedofs,:)=G(freedofs,freedofs)\F(freedofs,:);
    Obj(iter,1)=(F'*U)*conj(F'*U);
    %% SENSITIVITY
    ke1=reshape(sum((U(edofMat)*KE1).*U(edofMat),2),nely,nelx);
    ke2=reshape(sum((U(edofMat)*KE2).*U(edofMat),2),nely,nelx);
    dK=-(xPhys+(1-xPhys)*Emin).*(ke1-ke2);
    me1=reshape(sum((U(edofMat)*ME1).*U(edofMat),2),nely,nelx);
    me2=reshape(sum((U(edofMat)*ME2).*U(edofMat),2),nely,nelx);
    dM=-(xPhys+(1-xPhys)*Emin).*(me1-me2);
    dC=betac*dK+alphac*dM;
    udGu=dK+i*wt*dC-wt*wt*dM;
    dc=2*real(conj(F')*conj(U)*udGu);
    dm=ones(nely,nelx)/nelx/nely;
    %% FILTERING
    dc(:)=H*(dc(:)./Hs);
    dm(:)=H*(dm(:)./Hs);
    %% MMA OBJ % CONS
    f0val=scale_obj*Obj(end);
    df0dx=scale_obj*dc(:);
    fval(1,1)=scale_cons*(Con(end)/volfrac-1);
    dfdx(1,:)=scale_cons*dm(:)'/volfrac;
    %% MMA SOLVER
    dv=x(:); xmax=min(1,dv+move); xmin=max(0,dv-move);
    [xmma,~,~,~,~,~,~,~,~,low,upp]=...
        mmasub(m,n,iter,dv,xmin,xmax,xold1,xold2,f0val,df0dx,fval,dfdx,low,upp,a0,a,cmma,d);
    if iter>2, xold2=xold1; xold1=dv; elseif iter>1, xold1=dv; end
    xnew=reshape(xmma,nely,nelx);
    xPhys(:)=(H*xnew(:))./Hs;
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH/STEP FUNCTION
    xg=interp_xg(nelx,nely,xPhys,Hn,Hns,nodex,nodey,fnx,fny);
    [ls,xPhys,tol]=updatexPhys(nelx,nely,xg,ngrid,xPhys,beta);
    %% CHECK CONVERGENCE
    change=sum(abs(xnew(:)-x(:)))/(volfrac*nely*nelx);
    x=xnew;
    %% PLOT RESULTS
    fprintf('It.:%3i Obj.:%6.3f Vol.:%4.3f tol.:%4.3f chan.:%4.3f beta.:%g.\n',iter,Obj(end),Con(end),tol,change,beta);
    subplot(2,1,1); imagesc(-xPhys); axis equal tight off; drawnow
    subplot(2,1,2); contourf(fnx, flipud(fny), xg-ls, [0 0]); axis equal tight off; drawnow
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada     %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%